----------------
title : C++ 레퍼런스 - string 의 insert 함수
cat_title :  insert
ref_title : insert
path : /C++ Reference/string
publish_date : 2020-07-17
----------------

##@ cpp-ref-start

#@ insert

`insert` 함수는 문자열의 중간에 문자를 추가해주는 함수로, 편의를 위해 여러가지 형태의 함수들을 제공하고 있습니다.

```cpp-formatted
basic_string& insert(size_type index, size_type count, CharT ch);
basic_string& insert(size_type index, const CharT* s);
basic_string& insert(size_type index, const CharT* s, size_type count);
basic_string& insert(size_type index, const basic_string& str);
basic_string& insert(size_type index, const basic_string& str,
                     size_type index_str, size_type count = npos);
```

`index` 위치에 문자 혹은 문자열을 삽입한다. 인자로 `CharT` 를 받는 경우, 해당 문자를 `count` 개 만큼 삽입합니다. 인자로 `const CharT*` 나 `const basic_string&` 을 받는 경우 전달되는 문자열 (포인터의 경우 NULL 종료 문자열이라 생각) 을 `index` 위치에 삽입합니다.

참고로 `index` 위치에 집어넣는다는 뜻은, **삽입되는 문자열의 시작 위치가 index 번째**가 된다는 뜻입니다. 즉, 기존에 `index` 위치에 있던 문자는 삽입되는 문자열 맨 뒤로 밀리게 됩니다.

```cpp-formatted
iterator insert(const_iterator pos, CharT ch);
iterator insert(const_iterator pos, size_type count, CharT ch);
```

반복자 `pos` 가 가리키는 위치에 문자 `ch` 를 삽입합니다. `count` 를 지정하는 경우, `count` 개 만큼의 문자들을 삽입합니다.

```cpp-formatted
template <class InputIt>
iterator insert(const_iterator pos, InputIt first, InputIt last);
```

반복자 `pos` 가 가리키는 위치에 `first` 부터 `last` 전 까지의 문자열을 삽입합니다.

```cpp-formatted
iterator insert(const_iterator pos, std::initializer_list<CharT> ilist);
```

반복자 `pos` 가 가리키는 위치에, 초기화자 `ilist` 의 문자들을 삽입합니다.

```cpp-formatted
template <class T>
basic_string& insert(size_type pos, const T& t);
template <class T>
basic_string& insert(size_type index, const T& t, size_type index_str,
                     size_type count = npos);
```

t 를 `string_view sv` 로 변한한 뒤에 `pos` 가 가리키는 위치에 `sv` 의 원소들을 삽입합니다. 이는 마치,

```cpp-formatted
std::basic_string_view<CharT, Traits> sv = t;
insert(pos, sv.data(), sv.size())
```

와 동일하다고 볼 수 있습니다. 참고로 이를 위해서, 타입 T 가 `basic_string_view` 로 변환 가능한 타입이어야 합니다. 두 번째 경우도 위와 동일하지만, `sv` 전체 문자열이 아닌, `index_str` 부터 끝까지의 문자들을 삽입합니다.

### 인자

* `index`	-	문자열이 삽입될 위치 (정수)
* `pos`	- 문자열이 삽입될 위치 (반복자)
* `ch`	-	삽입할 문자
* `count`	-	삽입할 문자의 개수
* `s`	-	삽입할 문자열을 가리키는 포인터 (널 종료 문자열을 가리켜야 함!)
* `str`	-	삽입할 문자열
* `first, last`	-	삽입할 문자열의 시작과 끝을 가리킴 (last 직접 까지 삽입됨)
* `index_str`	-	삽입할 문자열의 시작점
* `ilist`	- 삽입할 초기화자 리스트 (`std::initializer_list`) 의 문자들
* `t` -	`std::basic_string_view` 로 변환 가능한 객체

### 리턴값

반복자를 사용하지 않는 경우, 삽입된 문자열이 리턴되고, 반복자를 사용하는 경우, 삽입된 위치를 가리키는 반복자를 리턴한다.

참고적으로, 문자열을 삽입하는 위치가 문자열 범위를 벗어난다면 `std::out_of_range` 예외를 발생시킨다. 만약에, 문자열을 삽입하는 위치는 정상적이지만, 삽입되는 문자열이 너무 길다면 (C++ 문자열의 최대 길이는 `str.max_size()` 로 정의됨) `std::length_error` 을 발생 시킨다.

```cpp-formatted
#include <cassert>
#include <iterator>
#include <string>
using namespace std::string_literals;
int main() {
  std::string s = "xmplr";

  // insert(size_type index, size_type count, char ch)
  s.insert(0, 1, 'E');
  assert("Exmplr" == s);

  // insert(size_type index, const char* s)
  s.insert(2, "e");
  assert("Exemplr" == s);

  // insert(size_type index, string const& str)
  s.insert(6, "a"s);
  assert("Exemplar" == s);

  // insert(size_type index, string const& str,
  //     size_type index_str, size_type count)
  s.insert(8, " is an example string."s, 0, 14);
  assert("Exemplar is an example" == s);

  // insert(const_iterator pos, char ch)
  s.insert(s.cbegin() + s.find_first_of('n') + 1, ':');
  assert("Exemplar is an: example" == s);

  // insert(const_iterator pos, size_type count, char ch)
  s.insert(s.cbegin() + s.find_first_of(':') + 1, 2, '=');
  assert("Exemplar is an:== example" == s);

  // insert(const_iterator pos, InputIt first, InputIt last)
  {
    std::string seq = " string";
    s.insert(s.begin() + s.find_last_of('e') + 1, std::begin(seq),
             std::end(seq));
    assert("Exemplar is an:== example string" == s);
  }

  // insert(const_iterator pos, std::initializer_list<char>)
  s.insert(s.cbegin() + s.find_first_of('g') + 1, {'.'});
  assert("Exemplar is an:== example string." == s);
}
```

`assert` 문을 확인하면 결과를 알 수 있다.

### 참고 자료

* `append` - 문자열을 맨 뒤에 추가한다.
* `push_back` - 문자를 맨 뒤에 추가한다.